//=============================================================================
// MPP_MaskFog.js
//=============================================================================
// Copyright (c) 2022 Mokusei Penguin
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc Displays a partially transparent image in front of the map.
 * @author Mokusei Penguin
 * @url 
 *
 * @help [version 1.0.0]
 * This plugin is for RPG Maker MZ.
 * 
 * ▼ Plugin command
 *  - In the field where you enter a number, select the text and write v[N]
 *    to refer to the variable number N.
 *  
 *  〇 setMaskFogCavity
 *   - Make a circular transparency around the character.
 *   
 *  〇 moveMaskFogOpacity
 *   - Moves the opacity of the fog image over time.
 * 
 * ▼ Map notes
 *  〇 <MaskFog:xxx>
 *      xxx : Fog image name
 *   - Displays a fog image in front of the map.
 *   - Images are loaded from the img/parallaxes folder.
 *   - As with the distant view, adding "!" to the beginning of
 *     the image file name will set the parallax to 0.
 * 
 *  〇 <MaskFogScroll:x,y>
 *      x : X-axis scroll speed
 *      y : Y-axis scroll speed
 *   - The fog image scrolls automatically.
 *   - The movement speed and direction are the same as the distant view
 *     of the map.
 * 
 *  〇 <MaskFogOpacity:o>
 *      o : Opacity
 *   - Specifies the opacity of the fog image.
 * 
 * ▼ Annotation of event execution content
 *  〇 FogCavity r o
 *      r : radius (1 = 1 tile)
 *      o : Opacity (Set from 0-255)
 *   - Transmits the fog image circularly around this event.
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠ is half-width)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 * 
 *  @noteParam MaskFog
 *      @noteDir img/parallaxes/
 *      @noteType file
 *      @noteData maps
 * 
 * 
 *  @command setMaskFogCavity
 *      @desc 
 *      @arg character
 *          @desc -1:Player, 0:This event, 1..:EventID
 *          @type number
 *              @min -1
 *              @max 999
 *          @default 0
 *      @arg radius
 *          @desc 1 = 1 tile
 *          @type number
 *              @min 0
 *              @max 99
 *              @decimals 1
 *          @default 3.0
 *      @arg opacity
 *          @desc 
 *          @type number
 *              @min 0
 *              @max 255
 *          @default 255
 * 
 *  @command moveMaskFogOpacity
 *      @desc 
 *      @arg opacity
 *          @desc 
 *          @type number
 *              @min 0
 *              @max 255
 *          @default 255
 *      @arg duration
 *          @desc Specify the time to move in frames
 *          @type number
 *              @min 1
 *              @max 999
 *          @default 60
 * 
 * 
 *  @param Player Defaults
 *      @desc 
 *      @type struct<PlayerDefaults>
 *      @default {"Radius":"3.0","Opacity":"255"}
 *
 *  @param Circle Blur
 *      @desc 0～100
 *      @type number
 *          @min 0
 *          @max 100
 *      @default 50
 *
 */

/*~struct~PlayerDefaults:
 *  @param Radius
 *      @desc 1 = 1 tile
 *      @type number
 *          @min 0
 *          @max 99
 *          @decimals 1
 *      @default 3.0
 * 
 *   @param Opacity
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 255
 *      @default 255
 * 
 */

/*:ja
 * @target MZ
 * @plugindesc マップの前面に部分的に透過できる画像を表示します。
 * @author 木星ペンギン
 * @url 
 *
 * @help [version 1.0.0]
 * このプラグインはRPGツクールMZ用です。
 * 
 * ▼ プラグインコマンド
 *  - 数値を入力する項目で、テキストを選択して v[N] と記述することで
 *    変数N番を参照します。
 *  
 *  〇 フォグ透過設定
 *   - キャラクターを中心に円形の透過を行います。
 *   
 *  〇 フォグ不透明度の移動
 *   - フォグ画像の不透明度を時間をかけて移動します。
 * 
 * ▼ マップのメモ
 *  〇 <MaskFog:xxx>
 *      xxx : フォグ画像名
 *   - マップの前面にフォグ画像を表示します。
 *   - 画像は img/parallaxes フォルダから読み込まれます。
 *   - 遠景と同じく、画像ファイル名の先頭に「!」をつけると視差は0になります。
 * 
 *  〇 <MaskFogScroll:x,y>
 *      x : X軸のスクロール速度
 *      y : Y軸のスクロール速度
 *   - フォグ画像が自動でスクロールします。
 *   - 移動速度や方向はマップの遠景と同じです。
 * 
 *  〇 <MaskFogOpacity:o>
 *      o : 不透明度
 *   - フォグ画像の不透明度を指定します。
 * 
 * ▼ イベントの実行内容の注釈
 *  〇 フォグ透過 r o
 *      r : 半径(1 = 1タイル)
 *      o : 不透明度(0～255で設定)
 *   - このイベントを中心にフォグ画像を円形に透過します。
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠は半角)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 * 
 *  @noteParam MaskFog
 *      @noteDir img/parallaxes/
 *      @noteType file
 *      @noteData maps
 * 
 * 
 *  @command setMaskFogCavity
 *      @text フォグ透過設定
 *      @desc 
 *      @arg character
 *          @text 対象キャラクター
 *          @desc -1:プレイヤー, 0:このイベント, 1～:イベントID
 *          @type number
 *              @min -1
 *              @max 999
 *          @default 0
 *      @arg radius
 *          @text 半径
 *          @desc 1 = 1タイル
 *          @type number
 *              @min 0
 *              @max 99
 *              @decimals 1
 *          @default 3.0
 *      @arg opacity
 *          @text 不透明度
 *          @desc 
 *          @type number
 *              @min 0
 *              @max 255
 *          @default 255
 * 
 *  @command moveMaskFogOpacity
 *      @text フォグ不透明度の移動
 *      @desc 
 *      @arg opacity
 *          @text 不透明度
 *          @desc 
 *          @type number
 *              @min 0
 *              @max 255
 *          @default 255
 *      @arg duration
 *          @text 時間
 *          @desc 移動にかける時間をフレーム数で指定
 *          @type number
 *              @min 1
 *              @max 999
 *          @default 60
 * 
 * 
 *  @param Player Defaults
 *      @text プレイヤーのデフォルト値
 *      @desc 
 *      @type struct<PlayerDefaults>
 *      @default {"Radius":"3.0","Opacity":"255"}
 *
 *  @param Circle Blur
 *      @text ぼかし
 *      @desc 0～100
 *      @type number
 *          @min 0
 *          @max 100
 *      @default 50
 *
 */

/*~struct~PlayerDefaults:ja
 *  @param Radius
 *      @text 半径
 *      @desc 1 = 1タイル
 *      @type number
 *          @min 0
 *          @max 99
 *          @decimals 1
 *      @default 3.0
 * 
 *   @param Opacity
 *      @text 不透明度
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 255
 *      @default 255
 * 
 */

(() => {
    'use strict';
    
    const pluginName = 'MPP_MaskFog';

    // Plugin Parameters
    const parameters = PluginManager.parameters(pluginName);
    const paramReplace = (key, value) => {
        try {
            return JSON.parse(value);
        } catch (e) {
            return value;
        }
    };
    const param_PlayerDefaults = JSON.parse(parameters['Player Defaults'] || '{}', paramReplace);
    const param_CircleBlur = Number(parameters['Circle Blur'] || 50);

    /**
     * JsExtensions の代替。指定された範囲に値が制限されている数値を返します。
     * Number.prototype.clamp と違い、下限優先。
     * 
     * @param {number} x - 値。
     * @param {number} min - 下限。
     * @param {number} max - 上限。
     * @returns {number} 範囲内の数値。
     */
    const number_clamp = (x, min, max) => {
        return Math.max(Math.min(x, max), min);
    };

    //-------------------------------------------------------------------------
    // Tilemap

    const _Tilemap_destroy = Tilemap.prototype.destroy;
    Tilemap.prototype.destroy = function() {
        if (this._fogMaskSprite) {
            this._fogMaskSprite.destroy();
        }
        _Tilemap_destroy.apply(this, arguments);
    };

    Tilemap.prototype.createFogMaskSprite = function() {
        const m = this._margin;
        const width = this.width + this._tileWidth + m * 2;
        const height = this.height + this._tileHeight + m * 2;
        this._fogMaskSprite = new Sprite_FogMask(width, height);
        this._fogMaskSprite.setOrigin(-m, -m);
    };

    Tilemap.prototype.fogMaskSprite = function() {
        return this._fogMaskSprite;
    };
    
    const _Tilemap_update = Tilemap.prototype.update;
    Tilemap.prototype.update = function() {
        _Tilemap_update.apply(this, arguments);
        this.updateFogMask();
    };
    
    Tilemap.prototype.updateFogMask = function() {
        if (this._fogMaskSprite) {
            this._fogMaskSprite.update();
        }
    };
    
    const _Tilemap_updateTransform = Tilemap.prototype.updateTransform;
    Tilemap.prototype.updateTransform = function() {
        _Tilemap_updateTransform.apply(this, arguments);
        if (this._fogMaskSprite) {
            this._fogMaskSprite.updateTransform();
        }
    };

    //-------------------------------------------------------------------------
    // Sprite_FogMask

    class Sprite_FogMask extends PIXI.Sprite {
        constructor(width, height) {
            const texture = PIXI.RenderTexture.create(width, height); 
            super(texture);
            this._container = new PIXI.Container();
            this._fogCavities = new Map();
            this._width = width;
            this._height = height;
            this.createBackGraphics();
            this.clearBack();
        }

        destroy() {
            const options = { children: true, texture: true };
            this._fogCavities = null;
            this._container.destroy(options);
            super.destroy(options);
        }

        createBackGraphics() {
            this._backGraphics = new PIXI.Graphics();
            this._container.addChild(this._backGraphics);
        }

        clearBack() {
            const { width, height } = this.texture;
            this._backGraphics.clear();
            this._backGraphics.beginFill(0xFFFFFF);
            this._backGraphics.drawRect(0, 0, width, height);
            this._backGraphics.endFill();
        }

        setOrigin(x, y) {
            this._backGraphics.x = x;
            this._backGraphics.y = y;
        }

        update() {
            const characters = $gameMap.allMaskFogCharacters();
            if (characters.length !== this._fogCavities.size) {
                for (const character of characters) {
                    if (!this._fogCavities.has(character)) {
                        const sprite = new Sprite_FogCavity(character);
                        this._fogCavities.set(character, sprite);
                        this._container.addChild(sprite);
                    }
                }
                for (const character of this._fogCavities.keys()) {
                    if (!characters.includes(character)) {
                        this._fogCavities.get(character).destroy();
                        this._fogCavities.delete(character);
                        this._container.removeChild(sprite);
                    }
                }
            }
            for (const sprite of this._fogCavities.values()) {
                sprite.update();
            }
        }
        
        updateTransform() {
            const renderer = Graphics.app.renderer;
            renderer.render(this._container, this.texture);
        }

        // Option1

        setBackBlur(strength) {
            this._backGraphics.filters = [new PIXI.filters.BlurFilter(strength)];
        }

        paintSpot(x, y, w, h, alpha) {
            const r = Math.floor(255 * (1 - alpha));
            this._backGraphics.beginFill(r * 256 * 256);
            this._backGraphics.drawRect(x, y, w, h);
            this._backGraphics.endFill();
        }

    }

    //-------------------------------------------------------------------------
    // Sprite_FogCavity

    class Sprite_FogCavity extends Sprite {
        constructor(character) {
            super();
            this.blendMode = PIXI.BLEND_MODES.LIGHTEN;
            this.opacity = 0;
            this.anchor.set(0.5);
            this._character = character;
            this._radius = 0;
            this._baseX = 0;
            this._baseY = 0;
            this._targetRadius = 0;
            this._targetOpacity = 0;
        }

        static _circleBitmaps = null;

        static circleBitmap(r) {
            if (r === 0) {
                return null;
            }
            this._circleBitmaps = this._circleBitmaps || {};
            if (r in this._circleBitmaps) {
                return this._circleBitmaps[r];
            }
            console.log(r);
            const bitmap = new Bitmap(r * 2, r * 2);
            let color;
            if (param_CircleBlur > 0) {
                const b = number_clamp(100 - param_CircleBlur, 0, 99) / 100;
                color = bitmap.context.createRadialGradient(r, r, 0, r, r, r);
                color.addColorStop(0, 'black');
                color.addColorStop(b, 'black');
                color.addColorStop(1, 'rgba(0,0,0,0)');
            } else {
                color = 'black';
            }
            bitmap.drawCircle(r, r, r, color);
            this._circleBitmaps[r] = bitmap;
            return bitmap;
        }


        destroy() {
            const cache = Sprite_FogCavity._circleBitmaps;
            if (cache) {
                for (const r in cache) {
                    cache[r].destroy();
                }
                Sprite_FogCavity._circleBitmaps = null;
            }
            super.destroy();
        }

        delete() {
            this._baseX = 0;
            this._baseY = 0;
            this._targetOpacity = 0;
        }

        start(x, y, r, opacity) {
            this._baseX = x;
            this._baseY = y;
            this._targetRadius = r;
            this._targetOpacity = opacity;
        }

        targetRadius() {
            return this._character ? this._character.fogCavityRadius() : this._targetRadius;
        }

        targetOpacity() {
            return this._character ? this._character.fogCavityOpacity() : this._targetOpacity;
        }

        update() {
            super.update();
            this.updateBitmap();
            this.updatePosition();
            this.updateOpacity();
            this.updateVisibility();
        }

        updateVisibility() {
            super.updateVisibility();
            if (this._character && this._character.isTransparent()) {
                this.visible = false;
            }
        }
        
        updateBitmap() {
            const radius = this.targetRadius();
            if (this._radius !== radius) {
                const r = Math.floor(radius * ($gameMap.tileWidth() + $gameMap.tileHeight()) / 2);
                this._radius = radius;
                this.bitmap = Sprite_FogCavity.circleBitmap(r);
                this.opacity = this.targetOpacity();
            }
        }

        updatePosition() {
            if (this._character) {
                this.x = this._character.screenX();
                this.y = this._character.screenY() - $gameMap.tileHeight() / 2;
            } else {
                this.x = $gameMap.adjustX(this._baseX);
                this.y = $gameMap.adjustY(this._baseY - 0.5);
            }
        }

        updateOpacity() {
            const opacity = this.targetOpacity();
            if (this.opacity < opacity) {
                this.opacity = Math.min(this.opacity + 8, opacity);
            } else if (this.opacity > opacity) {
                this.opacity = Math.max(this.opacity - 8, opacity);
            }
        }

    }

    //-------------------------------------------------------------------------
    // Game_Map

    const _Game_Map_setup = Game_Map.prototype.setup;
    Game_Map.prototype.setup = function(mapId) {
        _Game_Map_setup.call(this, mapId);
        this._maskFogName = '';
        this._maskFogX = 0;
        this._maskFogY = 0;
        this._maskFogSx = 0;
        this._maskFogSy = 0;
        this._maskFogOpacity = 0;
        this._targetMaskFogOpacity = 0;
        this._maskFogOpacityDuration = 0;
        if ($dataMap) {
            this.setupMaskFog();
        }
    };
    
    Game_Map.prototype.setupMaskFog = function() {
        const meta = $dataMap.meta;
        this._maskFogName = meta.MaskFog || '';
        this._maskFogZero = ImageManager.isZeroParallax(this._maskFogName);
        if ('MaskFogScroll' in meta) {
            const [sx = 0, sy = 0] = meta.MaskFogScroll.split(',').map(Number);
            this._maskFogSx = sx;
            this._maskFogSy = sy;
        } else {
            this._maskFogSx = 0;
            this._maskFogSy = 0;
        }
        if (this.isMaskFog()) {
            this._maskFogOpacity = 'MaskFogOpacity' in meta
                ? Number(meta.MaskFogOpacity || 0)
                : 255;
        }
    };

    const _Game_Map_setDisplayPos = Game_Map.prototype.setDisplayPos;
    Game_Map.prototype.setDisplayPos = function(x, y) {
        _Game_Map_setDisplayPos.apply(this, arguments);
        this._maskFogX = this.isLoopHorizontal() ? x : this._displayX;
        this._maskFogY = this.isLoopVertical() ? y : this._displayY;
    };
    
    Game_Map.prototype.isMaskFog = function() {
        return this._maskFogName !== '';
    };
    
    Game_Map.prototype.maskFogName = function() {
        return this._maskFogName;
    };
    
    Game_Map.prototype.maskFogOx = function() {
        return this._maskFogZero
            ? this._maskFogX * this.tileWidth()
            : (this._maskFogX * this.tileWidth()) / 2;
    };
    
    Game_Map.prototype.maskFogOy = function() {
        return this._maskFogZero
            ? this._maskFogY * this.tileHeight()
            : (this._maskFogY * this.tileHeight()) / 2;
    };
    
    Game_Map.prototype.maskFogOpacity = function() {
        return this._maskFogOpacity;
    };
    
    const _Game_Map_scrollDown = Game_Map.prototype.scrollDown;
    Game_Map.prototype.scrollDown = function(distance) {
        const lastY = this._displayY;
        _Game_Map_scrollDown.apply(this, arguments);
        if (this.isLoopVertical()) {
            this._maskFogY += distance;
        } else if (this.height() >= this.screenTileY()) {
            this._maskFogY += this._displayY - lastY;
        }
    };
    
    const _Game_Map_scrollLeft = Game_Map.prototype.scrollLeft;
    Game_Map.prototype.scrollLeft = function(distance) {
        const lastX = this._displayX;
        _Game_Map_scrollLeft.apply(this, arguments);
        if (this.isLoopHorizontal()) {
            this._maskFogX -= distance;
        } else if (this.width() >= this.screenTileX()) {
            this._maskFogX += this._displayX - lastX;
        }
    };
    
    const _Game_Map_scrollRight = Game_Map.prototype.scrollRight;
    Game_Map.prototype.scrollRight = function(distance) {
        const lastX = this._displayX;
        _Game_Map_scrollRight.apply(this, arguments);
        if (this.isLoopHorizontal()) {
            this._maskFogX += distance;
        } else if (this.width() >= this.screenTileX()) {
            this._maskFogX += this._displayX - lastX;
        }
    };
    
    const _Game_Map_scrollUp = Game_Map.prototype.scrollUp;
    Game_Map.prototype.scrollUp = function(distance) {
        const lastY = this._displayY;
        _Game_Map_scrollUp.apply(this, arguments);
        if (this.isLoopVertical()) {
            this._maskFogY -= distance;
        } else if (this.height() >= this.screenTileY()) {
            this._maskFogY += this._displayY - lastY;
        }
    };
    
    Game_Map.prototype.moveMaskFogOpacity = function(opacity, duration) {
        this._targetMaskFogOpacity = opacity;
        this._maskFogOpacityDuration = duration;
    };
    
    const _Game_Map_update = Game_Map.prototype.update;
    Game_Map.prototype.update = function(sceneActive) {
        _Game_Map_update.apply(this, arguments);
        this.updateMaskFog();
    };
    
    Game_Map.prototype.updateMaskFog = function() {
        this._maskFogX += this._maskFogSx / this.tileWidth() / 2;
        this._maskFogY += this._maskFogSy / this.tileHeight() / 2;
        if (this._maskFogOpacityDuration > 0) {
            const d = this._maskFogOpacityDuration;
            const t = this._targetMaskFogOpacity;
            this._maskFogOpacity = (this._maskFogOpacity * (d - 1) + t) / d;
            this._maskFogOpacityDuration--;
        }
    };
    
    Game_Map.prototype.setMaskFogOpacity = function(opacity) {
        this._maskFogOpacity = opacity;
    };

    Game_Map.prototype.allMaskFogCharacters = function() {
        return [...this.events(), $gamePlayer];
    };
    
    //-------------------------------------------------------------------------
    // Game_Character

    const _Game_Character_initMembers = Game_Character.prototype.initMembers;
    Game_Character.prototype.initMembers = function() {
        _Game_Character_initMembers.apply(this, arguments);
        this.clearFogCavity();
    };
    
    Game_Character.prototype.fogCavityRadius = function() {
        return this._fogCavityRadius;
    };
    
    Game_Character.prototype.fogCavityOpacity = function() {
        return this._fogCavityOpacity;
    };
    
    Game_Character.prototype.clearFogCavity = function() {
        this._fogCavityRadius = 0;
        this._fogCavityOpacity = 0;
    };
    
    Game_Character.prototype.deleteFogCavity = function() {
        this._fogCavityOpacity = 0;
    };
    
    Game_Character.prototype.setFogCavity = function(r, o) {
        this._fogCavityRadius = r;
        this._fogCavityOpacity = o;
    };
    
    //-------------------------------------------------------------------------
    // Game_Player

    const _Game_Player_initMembers = Game_Player.prototype.initMembers;
    Game_Player.prototype.initMembers = function() {
        _Game_Player_initMembers.apply(this, arguments);
        this._fogCavityRadius = param_PlayerDefaults.Radius || 0;
        this._fogCavityOpacity = param_PlayerDefaults.Opacity || 0;
    };
    
    //-------------------------------------------------------------------------
    // Game_Event

    const _Game_Event_clearPageSettings = Game_Event.prototype.clearPageSettings;
    Game_Event.prototype.clearPageSettings = function() {
        _Game_Event_clearPageSettings.apply(this, arguments);
        this.clearFogCavity();
    };
    
    const _Game_Event_setupPageSettings = Game_Event.prototype.setupPageSettings;
    Game_Event.prototype.setupPageSettings = function() {
        _Game_Event_setupPageSettings.apply(this, arguments);
        if (!this.setupFogCavity()) {
            this.clearFogCavity();
        }
    };

    Game_Event.prototype.setupFogCavity = function() {
        const comment = this.getCommentCommand('FogCavity', 'フォグ透過');
        if (comment) {
            const [_, ...args] = comment.split(' ');
            this.setFogCavity(...args.map(Number));
            return true;
        }
        return false;
    };

    Game_Event.prototype.getCommentCommand = function(...commandNames) {
        for (const evCom of this.list()) {
            if (evCom.code === 108 || evCom.code === 408) {
                const comment = evCom.parameters[0];
                if (commandNames.some(name => comment.startsWith(name))) {
                    return comment;
                }
            } else {
                return null;
            }
        }
        return null;
    };

    //-------------------------------------------------------------------------
    // PluginManager
    
    PluginManager.registerCommand(pluginName, 'setMaskFogCavity', function(args) {
        const character = this.character(PluginManager.mppValue(args.character));
        const r = PluginManager.mppValue(args.radius);
        const o = PluginManager.mppValue(args.opacity);
        if (character) {
            character.setFogCavity(r, o);
        }
    });

    PluginManager.registerCommand(pluginName, 'moveMaskFogOpacity', function(args) {
        const o = PluginManager.mppValue(args.opacity);
        const d = PluginManager.mppValue(args.duration);
        $gameMap.moveMaskFogOpacity(o, d);
    });

    PluginManager.mppValue = function(value) {
        const match = /^V\[(\d+)\]$/i.exec(value);
        return match ? $gameVariables.value(+match[1]) : +value;
    };
    
    //-------------------------------------------------------------------------
    // Spriteset_Map

    const _Spriteset_Map_destroy = Spriteset_Map.prototype.destroy;
    Spriteset_Map.prototype.destroy = function(options) {
        if (this._maskingFog) {
            this._maskingFog.destroy();
        }
        _Spriteset_Map_destroy.apply(this, arguments);
    };
    
    const _Spriteset_Map_createLowerLayer = Spriteset_Map.prototype.createLowerLayer;
    Spriteset_Map.prototype.createLowerLayer = function() {
        _Spriteset_Map_createLowerLayer.call(this);
        this.createMaskFog();
    };
    
    const _Spriteset_Map_update = Spriteset_Map.prototype.update;
    Spriteset_Map.prototype.update = function() {
        _Spriteset_Map_update.apply(this, arguments);
        this.updateMaskFog();
    };
    
    Spriteset_Map.prototype.createMaskFog = function() {
        if ($gameMap.isMaskFog()) {
            this._tilemap.createFogMaskSprite();
            this._maskingFog = new TilingSprite();
            this._maskingFog.move(0, 0, Graphics.width, Graphics.height);
            this._maskingFog.bitmap = ImageManager.loadParallax($gameMap.maskFogName());
            this._maskingFog.z = 6.1;
            this._maskingFog.mask = this._tilemap.fogMaskSprite();
            this._tilemap.addChild(this._maskingFog);
        }
    };
    
    Spriteset_Map.prototype.updateMaskFog = function() {
        if (this._maskingFog) {
            bitmap = this._maskingFog.bitmap;
            this._maskingFog.origin.x = $gameMap.maskFogOx() % bitmap.width;
            this._maskingFog.origin.y = $gameMap.maskFogOy() % bitmap.height;
            this._maskingFog.opacity = $gameMap.maskFogOpacity();
        }
    };
    
})();
